/*! \file tlk_cert_api.h
**********************************************************************************
*Title:                         Discretix Certificate Handling Toolkit source file
*
* Filename:                     tlk_cert_api.h
*
* Project, Target, subsystem:   Toolkit, Certificate Handling
* 
* Created:                      29.01.2008
*
* Modified:                     29.01.2008
*
* \Author                       Ira Boguslavsky
*
* \Remarks
*           Copyright (C) 2008 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/
#ifndef _TLK_CERT_API_H_
#define _TLK_CERT_API_H_

	/*----------- Doxygen Main Page text -----------------------------------------*/
	/*! \mainpage Certificate Handling Toolkit API
	    <DIV>
			<h1>Certificate Handling Toolkit API Documentation</h1>
			<p>
				<img width="268" height="148" src="DxLogo.jpg">
			</p>
			<P class="ConfidentialNotice" style="TEXT-ALIGN: justify">
				This document may be used 
				in its complete form only and is solely for the use of Discretix employees and 
                authorized Discretix channels or customers. The material herein is proprietary 
                to Discretix Ltd., any unauthorized reproduction of any part thereof is 
                strictly prohibited. The contents of this document are believed to be accurate 
				at the time of distribution. Discretix reserves the right to alter this 
				information at any time without notice. The primary distribution media of this 
				document is soft copy.
			</P>
			</DIV>
	**/

#ifndef DX_NO_TLK_SCLK_SUPPORT
    #include "tlk_sclk_api_types.h"
#else
    #include "tlk_cert_sclk_api_stub_types.h"
#endif

#include "tlk_cert_types.h"
#ifdef __cplusplus
extern "C"
{
#endif

/*TLK_CERT_CertificateIdCompute*/
/**
* @brief     The certificate identifier will be calculated from the  public key extracted from the certificate. 
*            The ID will be generated by performing 160-bit SHA-1 hash of the value of the BIT STRING subjectPublicKey 
*            (excluding the tag, length, and number of unused bits). 
*
* @param certificate_ptr           [in]     - Pointer to the certificate
* @param certificateId             [out]    - Pointer to the calculated certificate identifier
*
* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
* @return TLK_CERT_RC_ERROR_API_CERT_ID_COMPUTE_FAILED              - Certificate ID compute failed because of CRYS_HASH problems
* @return TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                - X.509 Parser got unsupported certificate version
* @return TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES       - X.509 Parser got too many extended key usages
* @return TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION    - X.509 Parser got unrecognized critical extension
* @return TLK_CERT_RC_ERROR_X509_INVALID_TIME                       - X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)   
* @return TLK_CERT_RC_ERROR_ASN1_BAD_ARGUMENTS                      - ASN1 Parser got invalid arguments
* @return TLK_CERT_RC_ERROR_ASN1_ITEM_NOT_FOUND                     - ASN1 Parser supposed to get specific structure of the item
* @return TLK_CERT_RC_ERROR_ASN1_OVERFLOW                           - ASN1 Parser got invalid tag of one of the fields in the structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_CLASS              - ASN1 Parser got unexpected item class
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_ENCODING           - ASN1 Parser got unexpected item encoding
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_TAG                - ASN1 Parser got unexpected item tag
* @return TLK_CERT_RC_ERROR_ASN1_INVALID_SEQ_STRUCTURE              - ASN1 Parser got invalid sequence structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_OID                     - ASN1 Parser got unexpected object identifier
* @return TLK_CERT_RC_ERROR_BUFF_SEEK_FAILED                        - Buffer seek operation failed
* @return TLK_CERT_RC_ERROR_BUFF_BUFFER_IS_NOT_BIG_ENOUGH           - Buffer is not big enough
* @return TLK_CERT_RC_ERROR_BUFF_ITEM_NOT_FOUND                     - Searching in buffer failed
* @return TLK_CERT_RC_ERROR_BUFF_NULL_POINTER                       - Null pointer received as input
* @return TLK_CERT_RC_ERROR_BUFF_NO_MORE_PLACE                      - The buffer capacity is over

*/
DxError_t TLK_CERT_CertificateIdCompute(                    const   TLK_CERT_Buffer_t           *certificate_ptr, 
                                                                    DxSha1Result_t               certificateId);

/**
* @brief     The purpose of this API is to retrieving all certificate parameters (field values).
*
* @param certificate_ptr           [in]     - Pointer to the certificate
* @param certificateParamList_ptr  [out]    - Pointer to the required certificate parameters list
*
* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
* @return TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                - X.509 Parser got unsupported certificate version
* @return TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES       - X.509 Parser got too many extended key usages
* @return TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION    - X.509 Parser got unrecognized critical extension
* @return TLK_CERT_RC_ERROR_X509_INVALID_TIME                       - X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)   
* @return TLK_CERT_RC_ERROR_ASN1_BAD_ARGUMENTS                      - ASN1 Parser got invalid arguments
* @return TLK_CERT_RC_ERROR_ASN1_ITEM_NOT_FOUND                     - ASN1 Parser supposed to get specific structure of the item
* @return TLK_CERT_RC_ERROR_ASN1_OVERFLOW                           - ASN1 Parser got invalid tag of one of the fields in the structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_CLASS              - ASN1 Parser got unexpected item class
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_ENCODING           - ASN1 Parser got unexpected item encoding
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_TAG                - ASN1 Parser got unexpected item tag
* @return TLK_CERT_RC_ERROR_ASN1_INVALID_SEQ_STRUCTURE              - ASN1 Parser got invalid sequence structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_OID                     - ASN1 Parser got unexpected object identifier
* @return TLK_CERT_RC_ERROR_BUFF_SEEK_FAILED                        - Buffer seek operation failed
* @return TLK_CERT_RC_ERROR_BUFF_BUFFER_IS_NOT_BIG_ENOUGH           - Buffer is not big enough
* @return TLK_CERT_RC_ERROR_BUFF_ITEM_NOT_FOUND                     - Searching in buffer failed
* @return TLK_CERT_RC_ERROR_BUFF_NULL_POINTER                       - Null pointer received as input
* @return TLK_CERT_RC_ERROR_BUFF_NO_MORE_PLACE                      - The buffer capacity is over
*/
DxError_t TLK_CERT_CertificateParamGet(                     const   TLK_CERT_Buffer_t           *certificate_ptr, 
                                                                    TLK_CERT_CertParameters_t   *certificateParamList_ptr);

/**
* @brief    This function initializes context object. 
*           The context object will be set at a starting position and it will be marked as just initialized.
* @note     User is responsible to provide of this context object.
*
* @param contextObject_ptr      [out]        - Pointer to the calculated certificate identifier
*
* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
* @return TLK_CERT_RC_ERROR_API_CONTEXT_OBJ_MAC_CALCULATE_FAILED    - Context Object MAC calculation failed because of CRYS_HASH problems
* @return TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                - X.509 Parser got unsupported certificate version
* @return TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES       - X.509 Parser got too many extended key usages
* @return TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION    - X.509 Parser got unrecognized critical extension
* @return TLK_CERT_RC_ERROR_X509_INVALID_TIME                       - X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)   
* @return TLK_CERT_RC_ERROR_ASN1_BAD_ARGUMENTS                      - ASN1 Parser got invalid arguments
* @return TLK_CERT_RC_ERROR_ASN1_ITEM_NOT_FOUND                     - ASN1 Parser supposed to get specific structure of the item
* @return TLK_CERT_RC_ERROR_ASN1_OVERFLOW                           - ASN1 Parser got invalid tag of one of the fields in the structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_CLASS              - ASN1 Parser got unexpected item class
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_ENCODING           - ASN1 Parser got unexpected item encoding
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_TAG                - ASN1 Parser got unexpected item tag
* @return TLK_CERT_RC_ERROR_ASN1_INVALID_SEQ_STRUCTURE              - ASN1 Parser got invalid sequence structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_OID                     - ASN1 Parser got unexpected object identifier
* @return TLK_CERT_RC_ERROR_BUFF_SEEK_FAILED                        - Buffer seek operation failed
* @return TLK_CERT_RC_ERROR_BUFF_BUFFER_IS_NOT_BIG_ENOUGH           - Buffer is not big enough
* @return TLK_CERT_RC_ERROR_BUFF_ITEM_NOT_FOUND                     - Searching in buffer failed
* @return TLK_CERT_RC_ERROR_BUFF_NULL_POINTER                       - Null pointer received as input
* @return TLK_CERT_RC_ERROR_BUFF_NO_MORE_PLACE                      - The buffer capacity is over
*/
DxError_t TLK_CERT_ChainValidationInit( TLK_CERT_ContextObj_t       *contextObject_ptr);

/**
* @brief    This a main function which covers all three steps of the chain validation algorithm:
*           Initialization, Processing and Termination. It will go through every 
*           field in the certificate structure and will perform validation according to the step and data 
*           which is stored in the context object. The user will be responsible to provide the context object, 
*           which will be an input for the every step. . (See requirement CVP_2_1).
*           Also this object will be protected cryptographically (MAC) or will be allocated in the 
*           trust environment. (See requirement CVP_2_2).
* @note     User is responsible for contextObject_ptr (It is MAC protected)
*
* @param prevCertificate_ptr       [in]      - Pointer to the previous certificate.
* @param currCertificate_ptr       [in]      - Pointer to the certificate to be processed.
* @param servClkDesc_ptr           [in]      - Service clock descriptor supplied by user. 
*                                              If user decides execute time validation checks under secure clock, 
*                                              he/she needs do the following:
*                                              * Add the TLK SCLK to his/her project.
*                                              * Set required configuration in config file
*                                              * Perform required secured clock initialization(Init, generate and re-validate)
*                                              * Pass to this service clock descriptor to the API
* @param servClkPwd_ptr            [in]      - Secured Clock password
* @param contextObject_ptr         [in/out]  - Pointer to the context object
* @param workspace_ptr             [in]      - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes      [in]      - Length of the workspace in bytes
*
* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
* @return TLK_CERT_RC_ERROR_API_CONTEXT_OBJ_MAC_CALCULATE_FAILED    - Context Object MAC calculation failed because of CRYS_HASH problems
* @return TLK_CERT_RC_ERROR_INTEGRITY                               - Context object integrity compromised
* @return TLK_CERT_RC_ERROR_X509_UTIL_NOT_BEFORE_TIME_FIELD_FAILED  - Problem to convert validNotBefore field from date format to seconds_sinse format
* @return TLK_CERT_RC_ERROR_X509_UTIL_NOT_AFTER_TIME_FIELD_FAILED   - Problem to convert validNotAfter field from date format to seconds_sinse format
* @return TLK_CERT_RC_ERROR_X509_UTIL_VALIDITY_PERIOD_VERIFY_FAILED - The current certificate is expired
* @return TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                - X.509 Parser got unsupported certificate version
* @return TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES       - X.509 Parser got too many extended key usages
* @return TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION    - X.509 Parser got unrecognized critical extension
* @return TLK_CERT_RC_ERROR_X509_INVALID_TIME                       - X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)   
* @return TLK_CERT_RC_ERROR_X509_UTIL_SUBJECT_NAME_INVALID          - Subject name of the current ("parent") certificate is differ from the issuer name of the "child" certificate
* @return TLK_CERT_RC_ERROR_X509_UTIL_SUBJECT_KEY_ID_INVALID        - Subject key identifier is invalid
* @return TLK_CERT_RC_ERROR_X509_UTIL_KEY_USAGE_INVALID             - Key usage format incorrect or bit 5 (Sign bit) is off
* @return TLK_CERT_RC_ERROR_X509_UTIL_WORKSPACE_SIZE_TOO_SMALL      - User supplied too small workspace size
* @return TLK_CERT_RC_ERROR_X509_UTIL_IS_CA_FLAG_INVALID            - Expected that basicConstraints CA flag is set
* @return TLK_CERT_RC_ERROR_X509_UTIL_EE_SUBJECT_NAME_MISSING       - Subject name in end-entity certificate missing
* @return TLK_CERT_RC_ERROR_X509_UTIL_TR_SUBJECT_NAME_MISSING       - Subject name in trusted root certificate missing
* @return TLK_CERT_RC_ERROR_X509_UTIL_SUBJECT_NAME_MISSING          - Subject name in certificate missing
* @return TLK_CERT_RC_ERROR_X509_UTIL_USSUER_NAME_MISSING           - Issuer name in certificate missing
* @return TLK_CERT_RC_ERROR_X509_UTIL_KEY_IDENTIFIER_VERIFY_FAILD   - Key Identifier verification failed
* @return TLK_CERT_RC_ERROR_ASN1_BAD_ARGUMENTS                      - ASN1 Parser got invalid arguments
* @return TLK_CERT_RC_ERROR_ASN1_ITEM_NOT_FOUND                     - ASN1 Parser supposed to get specific structure of the item
* @return TLK_CERT_RC_ERROR_ASN1_OVERFLOW                           - ASN1 Parser got invalid tag of one of the fields in the structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_CLASS              - ASN1 Parser got unexpected item class
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_ENCODING           - ASN1 Parser got unexpected item encoding
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_TAG                - ASN1 Parser got unexpected item tag
* @return TLK_CERT_RC_ERROR_ASN1_INVALID_SEQ_STRUCTURE              - ASN1 Parser got invalid sequence structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_OID                     - ASN1 Parser got unexpected object identifier
* @return TLK_CERT_RC_ERROR_BUFF_SEEK_FAILED                        - Buffer seek operation failed
* @return TLK_CERT_RC_ERROR_BUFF_BUFFER_IS_NOT_BIG_ENOUGH           - Buffer is not big enough
* @return TLK_CERT_RC_ERROR_BUFF_ITEM_NOT_FOUND                     - Searching in buffer failed
* @return TLK_CERT_RC_ERROR_BUFF_NULL_POINTER                       - Null pointer received as input
* @return TLK_CERT_RC_ERROR_BUFF_NO_MORE_PLACE                      - The buffer capacity is over
*/
DxError_t TLK_CERT_ChainValidationProcess(                  const   TLK_CERT_Buffer_t                   *prevCertificate_ptr,
                                                            const   TLK_CERT_Buffer_t                   *currCertificate_ptr, 
                                                            const   TLK_SCLK_ServiceClockDescriptor_t   *servClkDesc_ptr,
                                                                    TLK_CERT_Buffer_t                   *servClkPwd_ptr,
                                                                    TLK_CERT_ContextObj_t               *contextObject_ptr,
                                                                    DxUint8_t                           *workspace_ptr,
                                                            const   DxUint32_t                           workspaceSizeInBytes);
 
/**
* @brief    The CVPO will be filled from the context object. 
* @note     User is responsible for contextObject_ptr (It is MAC protected)
*
* @param cvpoObject_ptr         [out]        - Pointer to the certificate
* @param contextObject_ptr      [in]         - Pointer to the calculated certificate identifier
*
* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
*/
DxError_t TLK_CERT_ChainValidationTerminate(                        TLK_CERT_ContextObj_t       *contextObject_ptr,
                                                                    TLK_CERT_CPVO_t             *cvpoObject_ptr);

/**
* @brief    This function generates nonce for OCSP protocol for protection against replay attacks on the protocol.
* @note     The buffSizeInBytes field will be used as in/out parameter. This function will check the size of the input buffer, If 
*           it is not big enough, the function will return relevant error code and required buffer size. If the buffer size will be 
*           OK, the buffSizeInBytes field will updated to the actual one.
*
* @param nonce_ptr                 [in/out]  - Pointer to the nonce
* @param servClkDesc_ptr           [in/out]  - Service clock descriptor supplied by user. 
*                                              If user decides execute time validation checks under secure clock, 
*                                              he/she needs do the following:
*                                              * Add the TLK SCLK to his/her project.
*                                              * Set required configuration in config file
*                                              * Perform required secured clock initialization(Init, generate and re-validate)
*                                              * Pass this service clock descriptor to the API
* @param servClkPwd_ptr            [in]      - Secured Clock password
* @param workspace_ptr             [in]      - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes      [in]      - Length of the workspace in bytes

* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
* @return TLK_CERT_RC_ERROR_API_NONCE_PART_GENERATION_FAILED        - Random nonce part generation of nonce failed
* @return TLK_CERT_RC_ERROR_API_NONCE_MAC_CALCULATION_FAILED        - Nonce structure MAC calculation failed because of CRYS_HASH problems
*/
DxError_t TLK_CERT_NonceIssuance(                                   TLK_CERT_Nonce_t                     *nonce_ptr,
                                                                    TLK_SCLK_ServiceClockDescriptor_t    *servClkDesc_ptr,
                                                                    TLK_CERT_Buffer_t                    *servClkPwd_ptr,
                                                                    DxUint8_t                            *workspace_ptr,
                                                                    DxUint32_t                            workspaceSizeInBytes);
/**
* @brief    The goal of this function is to verify the validity of a certificate against an OCSP response.
*           It should also check the validity of the response itself.
*
*           The API will process all inputs (response its self, nonce, certificate to be checked and
*           CA certificate, that issued the certificate to be checked) and will validate the following:
*           	Responder certificate verification;
*           	Signature verification on the OCSP response;
*           	Nonce verification;
*           	Certificate Id verification;
*           	Reported status for the certificate validation
*
*           Response fields will be following (minimum):
*           	responseStatus;
*           	responseBytes: 
*               o	responseType (id-pkix-ocsp-basic)
*               o	response (tbsResponseData, signatureAlgorithm, signature, certs)
*
* @param response_ptr                [in]        - The response its self;
* @param nonce_ptr                   [in]        - The nonce (optional)
* @param certToBeChecked_ptr         [in]        - The certificate to be checked;
* @param certResponder_ptr           [in]        - The certificate fro the OCSP responder (optional)
* @param certCA_ptr                  [in]        - The CA certificate, that issued the certificate to be checked
* @param servClkDescNonceValid_ptr   [in]        - Service clock descriptor, result of TLK_CERT_NonceIssuance API 
*                                                   and will used checking nonce staleness only 
* @param servClkPwdNonceValid_ptr    [in]        - Password of Service clock descriptor, result of TLK_CERT_NonceIssuance API 
*                                                   and will used checking nonce staleness only
* @param servClkDesc_ptr             [in]        - Service clock descriptor supplied by user. 
*                                                  If user decides execute time validation checks under secure clock, 
*                                                  he/she needs do the following:
*                                                  * Add the TLK SCLK to his/her project.
*                                                  * Set required configuration in config file
*                                                  * Perform required secured clock initialization(Init, generate and re-validate)
*                                                  * Pass to this service clock descriptor to the API
* @param servClkPwd_ptr              [in]        - Secured Clock password
* @param rvo_ptr                     [out]       - The pointer to the response verify object
* @param workspace_ptr               [in]        - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes        [in]        - Length of the workspace in bytes

* @return TLK_CERT_RC_OK                                                - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                            - Null pointer received as input
* @return TLK_CERT_RC_ERROR_NO_SUPPLIED_CERTIFICATES                    - No certificate supplied by user
* @return TLK_CERT_RC_ERROR_CERTIFICATE_LENGTH_INVALID                  - The length of given certificate is incorrect
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_NO_SUPPLIED_CERTIFICATES          - There is no any supplied certificate
* @return TLK_CERT_RC_ERROR_API_CERT_IDENTITY_CALC_FAILED               - Failed on calculation of certificate identity because of CRYS_HASH problems
* @return TLK_CERT_RC_ERROR_API_RVO_MAC_CALCULATE_FAILED                - RVO MAC calculation failed because of CRYS_HASH problems
* @return TLK_CERT_RC_ERROR_OCSP_UNRECOGNIZED_CRITICAL_EXTENSION        - OCSP Parser successfully recognized the that extension is critical, but not kind of the extension
* @return TLK_CERT_RC_ERROR_OCSP_INVALID_SEQ_STRUCTURE                  - The structure of sequence is invalid
* @return TLK_CERT_RC_ERROR_OCSP_RESPONSE_IS_NOT_SUCCESSFULL            - The supplied response status is not successful
* @return TLK_CERT_RC_ERROR_OCSP_RESPONSE_UNSUPPORTED_TYPE              - The type of response is differ from Basic. We support only Basic OCSP Response types.
* @return TLK_CERT_RC_ERROR_OCSP_RESPONSE_UNSUPPORTED_VERSION           - The type of response id differ from V1.We support only v1 responses
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_FAIL                              - OCSP utility failed
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_ISSUER_NAME_INVALID               - The response issuer name is invalid
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_KEY_USAGE_EXTENSION_INVALID       - Key usage extension invalid (missing or invalid format)
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_INVALID_NUMBER_OF_CERTIFICATES    - The number of supplied certificates is over limited
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_NO_PROPER_RESPONDER_CERTIFICATE   - The response does not contains certificate relevant to supplied
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_TIME_VALIDATION_FAILED            - Certificate validity limit verification failed
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_NONCE_VALIDATION_FAILED           - Response Nonce validation failed
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_NO_SUPPLIED_CERTIFICATES          - There is no any supplied certificate
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_ISSUER_KEY_INVALID                - Issuer key (HASH) invalid
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_SERIAL_NUM_INVALID                - Serial number verification failed
* @return TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                    - X.509 Parser got unsupported certificate version
* @return TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES           - X.509 Parser got too many extended key usages
* @return TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION        - X.509 Parser got unrecognized critical extension
* @return TLK_CERT_RC_ERROR_X509_INVALID_TIME                           - X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)   
* @return TLK_CERT_RC_ERROR_ASN1_BAD_ARGUMENTS                          - ASN1 Parser got invalid arguments
* @return TLK_CERT_RC_ERROR_ASN1_ITEM_NOT_FOUND                         - ASN1 Parser supposed to get specific structure of the item
* @return TLK_CERT_RC_ERROR_ASN1_OVERFLOW                               - ASN1 Parser got invalid tag of one of the fields in the structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_CLASS                  - ASN1 Parser got unexpected item class
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_ENCODING               - ASN1 Parser got unexpected item encoding
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_TAG                    - ASN1 Parser got unexpected item tag
* @return TLK_CERT_RC_ERROR_ASN1_INVALID_SEQ_STRUCTURE                  - ASN1 Parser got invalid sequence structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_OID                         - ASN1 Parser got unexpected object identifier
* @return TLK_CERT_RC_ERROR_BUFF_SEEK_FAILED                            - Buffer seek operation failed
* @return TLK_CERT_RC_ERROR_BUFF_BUFFER_IS_NOT_BIG_ENOUGH               - Buffer is not big enough
* @return TLK_CERT_RC_ERROR_BUFF_ITEM_NOT_FOUND                         - Searching in buffer failed
* @return TLK_CERT_RC_ERROR_BUFF_NULL_POINTER                           - Null pointer received as input
* @return TLK_CERT_RC_ERROR_BUFF_NO_MORE_PLACE                          - The buffer capacity is over
*/
DxError_t TLK_CERT_ResponseProcess(                         const   TLK_CERT_Buffer_t                   *response_ptr,
                                                            const   TLK_CERT_Nonce_t                    *nonce_ptr,
                                                            const   TLK_CERT_Buffer_t                   *certToBeChecked_ptr,
                                                            const   TLK_CERT_Buffer_t                   *certResponder_ptr,
                                                            const   TLK_CERT_Buffer_t                   *certCA_ptr,
                                                                    TLK_SCLK_ServiceClockDescriptor_t   *servClkDescNonceValid_ptr,
                                                                    TLK_CERT_Buffer_t                   *servClkPwdNonceValid_ptr,
                                                                    TLK_SCLK_ServiceClockDescriptor_t   *servClkDesc_ptr,
                                                                    TLK_CERT_Buffer_t                   *servClkPwd_ptr,
                                                                    TLK_CERT_RV0_t                      *rvo_ptr,
                                                                    DxUint8_t                           *workspace_ptr,
                                                                    DxUint32_t                           workspaceSizeInBytes);

/**
* @brief    The purpose of this API is extraction responder information from the given OCSP response.
*           *   This function will check if the requested information exists in the given OCSP response (not NULL)
*           *   Extract the OCSP responder certificate 
*           *   Extract the response expiration time from the nextUpdate field.
*
* @param response_ptr           [in]         - The response, from which the certificate will be extracted
* @param cert_ptr               [in/out]     - Pointer to the certificate buffer
* @param exparationTime_ptr     [out]        - Pointer to the expiration time structure
*
* @return TLK_CERT_RC_OK                                                - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                            - Null pointer received as input.
* @return TLK_CERT_RC_ERROR_OCSP_UNRECOGNIZED_CRITICAL_EXTENSION        - OCSP Parser successfully recognized the that extension is critical, but not kind of the extension
* @return TLK_CERT_RC_ERROR_OCSP_INVALID_SEQ_STRUCTURE                  - The structure of sequence is invalid
* @return TLK_CERT_RC_ERROR_OCSP_RESPONSE_IS_NOT_SUCCESSFULL            - The supplied response status is not successful
* @return TLK_CERT_RC_ERROR_OCSP_RESPONSE_UNSUPPORTED_TYPE              - The type of response is differ from Basic. We support only Basic OCSP Response types.
* @return TLK_CERT_RC_ERROR_OCSP_RESPONSE_UNSUPPORTED_VERSION           - The type of response id differ from V1.We support only v1 responses
* @return TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                    - X.509 Parser got unsupported certificate version
* @return TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES           - X.509 Parser got too many extended key usages
* @return TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION        - X.509 Parser got unrecognized critical extension
* @return TLK_CERT_RC_ERROR_X509_INVALID_TIME                           - X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)   
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_INVALID_NUMBER_OF_CERTIFICATES    - The number of supplied certificates is over limited
* @return TLK_CERT_RC_ERROR_OCSP_UTIL_NO_SUPPLIED_CERTIFICATES          - There is no any supplied certificate

*/
DxError_t TLK_CERT_CertFromResponseExtract(                 const   TLK_CERT_Buffer_t           *response_ptr,
                                                                    TLK_CERT_Buffer_t           *cert_ptr, 
                                                                    DxGeneralizedTime_t         *exparationTime_ptr);
/*TLK_CERT_WorkspaceSizeGet*/
/*!
\brief Get the minimum workspace size needed per Toolkit Certificate Handling operation.

@param workspaceMinSize_ptr         [out]    - Minimum workspace size needed

@return TLK_CERT_RC_OK                       - The operation completed successfully.
**/
DxError_t TLK_CERT_WorkspaceSizeGet(                               DxUint32_t                 *workspaceMinSize_ptr);
#ifdef __cplusplus
}
#endif /*extern "C"*/
#endif

